<?php
/**
 * User Login API
 */

// Define the root path of the project
define('ROOT_PATH', realpath(__DIR__ . '/../../../'));

require_once ROOT_PATH . '/backend/api/cors.php';
handleCORS();

require_once ROOT_PATH . '/backend/config/database.php';
require_once ROOT_PATH . '/backend/config/config.php';

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    sendResponse('error', 'Method not allowed', null, 405);
}

$data = json_decode(file_get_contents("php://input"));

// Validate required fields
if (empty($data->username) || empty($data->password)) {
    sendResponse('error', 'Username and password are required', null, 400);
}

try {
    $database = new Database();
    $conn = $database->getConnection();

    // Fetch user by username or email
    $query = "SELECT id, username, email, password_hash, full_name, profile_picture, role, is_active, email_verified 
              FROM users 
              WHERE (username = :username OR email = :username) AND is_active = 1";
    
    $stmt = $conn->prepare($query);
    $stmt->bindParam(':username', $data->username);
    $stmt->execute();

    if ($stmt->rowCount() === 0) {
        sendResponse('error', 'Invalid credentials', null, 401);
    }

    $user = $stmt->fetch(PDO::FETCH_ASSOC);

    // Verify password
    if (!password_verify($data->password, $user['password_hash'])) {
        sendResponse('error', 'Invalid credentials', null, 401);
    }

    // Generate session token
    $session_token = generateToken();
    $expires_at = date('Y-m-d H:i:s', time() + JWT_EXPIRATION_TIME);

    // Create session
    $session_query = "INSERT INTO user_sessions (user_id, session_token, ip_address, user_agent, expires_at) 
                      VALUES (:user_id, :session_token, :ip, :user_agent, :expires_at)";
    $session_stmt = $conn->prepare($session_query);
    $session_stmt->bindParam(':user_id', $user['id']);
    $session_stmt->bindParam(':session_token', $session_token);
    $session_stmt->bindParam(':ip', $_SERVER['REMOTE_ADDR']);
    $session_stmt->bindParam(':user_agent', $_SERVER['HTTP_USER_AGENT']);
    $session_stmt->bindParam(':expires_at', $expires_at);
    $session_stmt->execute();

    // Update last login
    $update_query = "UPDATE users SET last_login = CURRENT_TIMESTAMP WHERE id = :user_id";
    $update_stmt = $conn->prepare($update_query);
    $update_stmt->bindParam(':user_id', $user['id']);
    $update_stmt->execute();

    // Mark user as online
    $online_query = "INSERT INTO online_users (user_id) VALUES (:user_id) 
                     ON DUPLICATE KEY UPDATE last_activity = CURRENT_TIMESTAMP";
    $online_stmt = $conn->prepare($online_query);
    $online_stmt->bindParam(':user_id', $user['id']);
    $online_stmt->execute();

    // Log activity
    $log_query = "INSERT INTO activity_logs (user_id, activity_type, description, ip_address) 
                  VALUES (:user_id, 'login', 'User logged in', :ip)";
    $log_stmt = $conn->prepare($log_query);
    $log_stmt->bindParam(':user_id', $user['id']);
    $log_stmt->bindParam(':ip', $_SERVER['REMOTE_ADDR']);
    $log_stmt->execute();

    unset($user['password_hash']);

    sendResponse('success', 'Login successful', [
        'user' => $user,
        'token' => $session_token,
        'expires_at' => $expires_at
    ], 200);

} catch (PDOException $e) {
    sendResponse('error', 'Database error: ' . $e->getMessage(), null, 500);
}
?>
