<?php
/**
 * Chat Messages API - Get messages
 */

// Define the root path of the project
define('ROOT_PATH', realpath(__DIR__ . '/../../..'));

require_once ROOT_PATH . '/backend/api/cors.php';
handleCORS();

require_once ROOT_PATH . '/backend/config/database.php';
require_once ROOT_PATH . '/backend/config/config.php';
require_once ROOT_PATH . '/backend/middleware/auth.php';

$user = authenticate();

if ($_SERVER['REQUEST_METHOD'] !== 'GET') {
    sendResponse('error', 'Method not allowed', null, 405);
}

try {
    $database = new Database();
    $conn = $database->getConnection();

    $page = isset($_GET['page']) ? (int)$_GET['page'] : 1;
    $limit = MESSAGES_PER_PAGE;
    $offset = ($page - 1) * $limit;
    $before_id = isset($_GET['before']) ? (int)$_GET['before'] : null;

    $where_clause = "WHERE m.is_deleted = 0";
    if ($before_id) {
        $where_clause .= " AND m.id < :before_id";
    }

    $query = "SELECT 
                m.id,
                m.user_id,
                m.message_text,
                m.message_type,
                m.attachment_url,
                m.edited_at,
                m.created_at,
                u.username,
                u.full_name,
                u.profile_picture,
                u.role
              FROM messages m
              JOIN users u ON m.user_id = u.id
              $where_clause
              ORDER BY m.created_at DESC
              LIMIT :limit OFFSET :offset";
    
    $stmt = $conn->prepare($query);
    $stmt->bindParam(':limit', $limit, PDO::PARAM_INT);
    $stmt->bindParam(':offset', $offset, PDO::PARAM_INT);
    if ($before_id) {
        $stmt->bindParam(':before_id', $before_id, PDO::PARAM_INT);
    }
    $stmt->execute();

    $messages = $stmt->fetchAll(PDO::FETCH_ASSOC);

    // Get reactions for each message
    foreach ($messages as &$message) {
        $reaction_query = "SELECT mr.reaction_emoji as emoji, mr.user_id, u.username 
                          FROM message_reactions mr
                          JOIN users u ON mr.user_id = u.id
                          WHERE mr.message_id = :message_id";
        $reaction_stmt = $conn->prepare($reaction_query);
        $reaction_stmt->bindParam(':message_id', $message['id']);
        $reaction_stmt->execute();
        $message['reactions'] = $reaction_stmt->fetchAll(PDO::FETCH_ASSOC);
        $message['reaction_count'] = count($message['reactions']);
    }

    // Get total count
    $count_query = "SELECT COUNT(*) as total FROM messages WHERE is_deleted = 0";
    $count_stmt = $conn->query($count_query);
    $total = $count_stmt->fetch(PDO::FETCH_ASSOC)['total'];

    sendResponse('success', 'Messages retrieved', [
        'messages' => array_reverse($messages),
        'pagination' => [
            'page' => $page,
            'limit' => $limit,
            'total' => (int)$total,
            'total_pages' => ceil($total / $limit)
        ]
    ], 200);

} catch (PDOException $e) {
    sendResponse('error', 'Database error: ' . $e->getMessage(), null, 500);
}
?>
