<?php
/**
 * Send Chat Message API
 */

// Define the root path of the project
define('ROOT_PATH', realpath(__DIR__ . '/../../..'));

require_once ROOT_PATH . '/backend/api/cors.php';
handleCORS();

require_once ROOT_PATH . '/backend/config/database.php';
require_once ROOT_PATH . '/backend/config/config.php';
require_once ROOT_PATH . '/backend/middleware/auth.php';

$user = authenticate();

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    sendResponse('error', 'Method not allowed', null, 405);
}

$data = json_decode(file_get_contents("php://input"));

// Validate message - either text or attachment required
if ((empty($data->message_text) || trim($data->message_text) === '') && empty($data->attachment_url)) {
    sendResponse('error', 'Message text or attachment is required', null, 400);
}

try {
    $database = new Database();
    $conn = $database->getConnection();

    $message_type = $data->message_type ?? 'text';
    $attachment_url = $data->attachment_url ?? null;

    // Insert message
    $query = "INSERT INTO messages (user_id, message_text, message_type, attachment_url) 
              VALUES (:user_id, :message_text, :message_type, :attachment_url)";
    
    $stmt = $conn->prepare($query);
    $stmt->bindParam(':user_id', $user['id']);
    $stmt->bindParam(':message_text', sanitizeInput($data->message_text));
    $stmt->bindParam(':message_type', $message_type);
    $stmt->bindParam(':attachment_url', $attachment_url);
    
    if ($stmt->execute()) {
        $message_id = $conn->lastInsertId();

        // Get the complete message with user info
        $get_query = "SELECT 
                        m.id,
                        m.user_id,
                        m.message_text,
                        m.message_type,
                        m.attachment_url,
                        m.created_at,
                        u.username,
                        u.full_name,
                        u.profile_picture,
                        u.role
                      FROM messages m
                      JOIN users u ON m.user_id = u.id
                      WHERE m.id = :message_id";
        
        $get_stmt = $conn->prepare($get_query);
        $get_stmt->bindParam(':message_id', $message_id);
        $get_stmt->execute();
        
        $message = $get_stmt->fetch(PDO::FETCH_ASSOC);

        // Create notifications for @mentions
        if (preg_match_all('/@(\w+)/', $data->message_text, $mentions)) {
            foreach ($mentions[1] as $mentioned_username) {
                $notify_query = "INSERT INTO notifications (user_id, notification_type, title, content, reference_id)
                                SELECT id, 'mention', :title, :content, :ref_id
                                FROM users WHERE username = :username AND id != :sender_id";
                $notify_stmt = $conn->prepare($notify_query);
                $title = "Mentioned by " . $user['username'];
                $notify_stmt->bindParam(':title', $title);
                $notify_stmt->bindParam(':content', $data->message_text);
                $notify_stmt->bindParam(':ref_id', $message_id);
                $notify_stmt->bindParam(':username', $mentioned_username);
                $notify_stmt->bindParam(':sender_id', $user['id']);
                $notify_stmt->execute();
            }
        }

        sendResponse('success', 'Message sent', $message, 201);
    } else {
        sendResponse('error', 'Failed to send message', null, 500);
    }

} catch (PDOException $e) {
    sendResponse('error', 'Database error: ' . $e->getMessage(), null, 500);
}
?>
