<?php
/**
 * User Profile API
 */

// Define the root path of the project
define('ROOT_PATH', realpath(__DIR__ . '/../../..'));

require_once ROOT_PATH . '/backend/api/cors.php';
handleCORS();

require_once ROOT_PATH . '/backend/config/database.php';
require_once ROOT_PATH . '/backend/config/config.php';
require_once ROOT_PATH . '/backend/middleware/auth.php';

$user = authenticate();

if ($_SERVER['REQUEST_METHOD'] === 'GET') {
    // Get user profile
    $user_id = isset($_GET['user_id']) ? (int)$_GET['user_id'] : $user['id'];

    try {
        $database = new Database();
        $conn = $database->getConnection();

        $query = "SELECT 
                    u.id,
                    u.username,
                    u.email,
                    u.full_name,
                    u.phone,
                    u.profile_picture,
                    u.location,
                    u.supporter_id,
                    u.role,
                    u.created_at,
                    u.last_login,
                    (SELECT COUNT(*) FROM messages WHERE user_id = u.id AND is_deleted = 0) as message_count,
                    (SELECT last_activity FROM online_users WHERE user_id = u.id) as last_activity
                  FROM users u
                  WHERE u.id = :user_id AND u.is_active = 1";
        
        $stmt = $conn->prepare($query);
        $stmt->bindParam(':user_id', $user_id);
        $stmt->execute();

        if ($stmt->rowCount() === 0) {
            sendResponse('error', 'User not found', null, 404);
        }

        $profile = $stmt->fetch(PDO::FETCH_ASSOC);
        
        // Check if user is online (last activity within 5 minutes)
        $profile['is_online'] = false;
        if ($profile['last_activity']) {
            $last_activity = strtotime($profile['last_activity']);
            $profile['is_online'] = (time() - $last_activity) < 300;
        }

        sendResponse('success', 'Profile retrieved', $profile, 200);

    } catch (PDOException $e) {
        sendResponse('error', 'Database error: ' . $e->getMessage(), null, 500);
    }

} elseif ($_SERVER['REQUEST_METHOD'] === 'PUT') {
    // Update user profile
    $data = json_decode(file_get_contents("php://input"));

    try {
        $database = new Database();
        $conn = $database->getConnection();

        $updates = [];
        $params = [':user_id' => $user['id']];

        if (isset($data->full_name)) {
            $updates[] = "full_name = :full_name";
            $params[':full_name'] = sanitizeInput($data->full_name);
        }
        if (isset($data->phone)) {
            $updates[] = "phone = :phone";
            $params[':phone'] = sanitizeInput($data->phone);
        }
        if (isset($data->location)) {
            $updates[] = "location = :location";
            $params[':location'] = sanitizeInput($data->location);
        }
        if (isset($data->profile_picture)) {
            $updates[] = "profile_picture = :profile_picture";
            $params[':profile_picture'] = sanitizeInput($data->profile_picture);
        }

        if (empty($updates)) {
            sendResponse('error', 'No fields to update', null, 400);
        }

        $query = "UPDATE users SET " . implode(', ', $updates) . " WHERE id = :user_id";
        $stmt = $conn->prepare($query);
        
        foreach ($params as $key => $value) {
            $stmt->bindValue($key, $value);
        }

        if ($stmt->execute()) {
            sendResponse('success', 'Profile updated', null, 200);
        } else {
            sendResponse('error', 'Failed to update profile', null, 500);
        }

    } catch (PDOException $e) {
        sendResponse('error', 'Database error: ' . $e->getMessage(), null, 500);
    }

} else {
    sendResponse('error', 'Method not allowed', null, 405);
}
?>
